import { api } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface Alumni {
  id: string;
  nisn: string;
  nis: string;
  nama_lengkap: string;
  tempat_lahir: string;
  tanggal_lahir: string;
  jenis_kelamin: string;
  agama: string;
  alamat: string;
  no_telepon: string;
  tahun_lulus: number;
  melanjutkan_ke: string;
  pekerjaan: string;
  alamat_sekarang: string;
}

interface ListAlumniResponse {
  alumni: Alumni[];
  total: number;
}

interface ListAlumniParams {
  madrasah_id?: string;
  search?: string;
  tahun_lulus?: number;
  limit?: number;
  offset?: number;
}

// Lists alumni by madrasah
export const listAlumni = api<ListAlumniParams, ListAlumniResponse>(
  { expose: true, method: "GET", path: "/alumni" },
  async (params) => {
    const { madrasah_id, search = "", tahun_lulus, limit = 50, offset = 0 } = params;
    
    let whereClause = "";
    let searchParams: any[] = [];
    let paramIndex = 1;
    
    if (madrasah_id) {
      whereClause = `WHERE madrasah_id = $${paramIndex}`;
      searchParams.push(madrasah_id);
      paramIndex++;
    }
    
    if (search) {
      const searchCondition = `(nama_lengkap ILIKE $${paramIndex} OR nisn ILIKE $${paramIndex} OR nis ILIKE $${paramIndex})`;
      whereClause = whereClause ? `${whereClause} AND ${searchCondition}` : `WHERE ${searchCondition}`;
      searchParams.push(`%${search}%`);
      paramIndex++;
    }
    
    if (tahun_lulus) {
      const tahunCondition = `tahun_lulus = $${paramIndex}`;
      whereClause = whereClause ? `${whereClause} AND ${tahunCondition}` : `WHERE ${tahunCondition}`;
      searchParams.push(tahun_lulus);
      paramIndex++;
    }
    
    const alumniList = await db.rawQueryAll<Alumni>(
      `SELECT id, nisn, nis, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
              no_telepon, tahun_lulus, melanjutkan_ke, pekerjaan, alamat_sekarang
       FROM alumni ${whereClause} 
       ORDER BY tahun_lulus DESC, nama_lengkap 
       LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`,
      ...searchParams, limit, offset
    );
    
    const totalResult = await db.rawQueryRow<{ count: number }>(
      `SELECT COUNT(*) as count FROM alumni ${whereClause}`,
      ...searchParams
    );
    
    return {
      alumni: alumniList,
      total: totalResult?.count || 0
    };
  }
);
