# ADAM Deployment Guide

Panduan deployment untuk ADAM (Aplikasi Database Madrasah) ke berbagai platform.

## Prerequisites

- Node.js 18 atau lebih baru
- npm atau yarn
- Encore CLI
- Git

## Environment Variables

Buat file `.env` untuk konfigurasi environment:

```env
# Database
DATABASE_URL=postgresql://username:password@localhost:5432/adam_db

# Application
NODE_ENV=production
PORT=4000

# Security
JWT_SECRET=your-super-secret-jwt-key
ENCRYPTION_KEY=your-encryption-key

# Optional: External Services
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USER=your-email@gmail.com
SMTP_PASS=your-app-password
```

## Deployment Options

### 1. Encore Cloud (Recommended)

Encore Cloud adalah platform hosting resmi untuk aplikasi Encore.ts:

```bash
# Login ke Encore
encore auth login

# Deploy ke staging
encore deploy --env staging

# Deploy ke production
encore deploy --env production
```

### 2. Vercel

Deploy frontend ke Vercel dan backend ke platform lain:

```bash
# Install Vercel CLI
npm i -g vercel

# Deploy frontend
cd frontend
vercel --prod

# Set environment variables di Vercel dashboard
```

### 3. Railway

Deploy full-stack ke Railway:

```bash
# Install Railway CLI
npm install -g @railway/cli

# Login
railway login

# Initialize project
railway init

# Deploy
railway up
```

### 4. Docker Deployment

Buat `Dockerfile`:

```dockerfile
FROM node:18-alpine

WORKDIR /app

# Copy package files
COPY package*.json ./
RUN npm ci --only=production

# Copy source code
COPY . .

# Build application
RUN npm run build

# Expose port
EXPOSE 4000

# Start application
CMD ["npm", "start"]
```

Build dan run:

```bash
# Build image
docker build -t adam-madrasah .

# Run container
docker run -p 4000:4000 -e DATABASE_URL=your-db-url adam-madrasah
```

### 5. VPS/Server Manual

Deploy ke VPS atau server sendiri:

```bash
# Clone repository
git clone https://github.com/your-username/adam-madrasah.git
cd adam-madrasah

# Install dependencies
npm install

# Build application
npm run build

# Install PM2 for process management
npm install -g pm2

# Start with PM2
pm2 start ecosystem.config.js

# Setup nginx reverse proxy
sudo nano /etc/nginx/sites-available/adam
```

Konfigurasi nginx:

```nginx
server {
    listen 80;
    server_name your-domain.com;

    location / {
        proxy_pass http://localhost:4000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}
```

## Database Setup

### PostgreSQL Production

1. **Managed Database** (Recommended):
   - Supabase
   - Railway PostgreSQL
   - AWS RDS
   - Google Cloud SQL

2. **Self-hosted**:
   ```bash
   # Install PostgreSQL
   sudo apt update
   sudo apt install postgresql postgresql-contrib

   # Create database
   sudo -u postgres createdb adam_production

   # Create user
   sudo -u postgres createuser --interactive adam_user
   ```

### Migration

Jalankan migration di production:

```bash
# Set production database URL
export DATABASE_URL=your-production-db-url

# Run migrations
encore db migrate up
```

## SSL Certificate

### Let's Encrypt (Free)

```bash
# Install certbot
sudo apt install certbot python3-certbot-nginx

# Get certificate
sudo certbot --nginx -d your-domain.com

# Auto-renewal
sudo crontab -e
# Add: 0 12 * * * /usr/bin/certbot renew --quiet
```

## Monitoring & Logging

### PM2 Monitoring

```bash
# Monitor processes
pm2 monit

# View logs
pm2 logs

# Restart application
pm2 restart adam
```

### Log Management

Setup log rotation:

```bash
# Install logrotate
sudo apt install logrotate

# Configure log rotation
sudo nano /etc/logrotate.d/adam
```

## Backup Strategy

### Database Backup

```bash
# Create backup script
#!/bin/bash
DATE=$(date +%Y%m%d_%H%M%S)
pg_dump $DATABASE_URL > backup_$DATE.sql
aws s3 cp backup_$DATE.sql s3://your-backup-bucket/
rm backup_$DATE.sql
```

### Automated Backup

Setup cron job:

```bash
crontab -e
# Add: 0 2 * * * /path/to/backup-script.sh
```

## Performance Optimization

### Frontend

```bash
# Build with optimization
npm run build

# Serve static files with nginx
location /static/ {
    alias /path/to/build/static/;
    expires 1y;
    add_header Cache-Control "public, immutable";
}
```

### Backend

```bash
# Enable gzip compression
gzip on;
gzip_types text/plain text/css application/json application/javascript;

# Connection pooling
export DB_POOL_SIZE=20
export DB_POOL_TIMEOUT=30000
```

## Security Checklist

- [ ] HTTPS enabled
- [ ] Environment variables secured
- [ ] Database access restricted
- [ ] Regular security updates
- [ ] Backup encryption
- [ ] Access logs monitoring
- [ ] Rate limiting implemented
- [ ] Input validation enabled

## Troubleshooting

### Common Issues

1. **Database Connection Error**:
   ```bash
   # Check database URL
   echo $DATABASE_URL
   
   # Test connection
   psql $DATABASE_URL -c "SELECT 1;"
   ```

2. **Port Already in Use**:
   ```bash
   # Find process using port
   lsof -i :4000
   
   # Kill process
   kill -9 PID
   ```

3. **Memory Issues**:
   ```bash
   # Increase Node.js memory limit
   export NODE_OPTIONS="--max-old-space-size=4096"
   ```

### Health Checks

Setup health check endpoint:

```bash
# Check application health
curl http://localhost:4000/health

# Expected response: {"status": "ok", "timestamp": "..."}
```

## Rollback Strategy

### Quick Rollback

```bash
# Rollback to previous version
git checkout previous-stable-tag
npm install
npm run build
pm2 restart adam
```

### Database Rollback

```bash
# Rollback migration
encore db migrate down

# Restore from backup
psql $DATABASE_URL < backup_file.sql
```

## Support

Untuk bantuan deployment:

1. Check dokumentasi Encore.ts
2. Review GitHub Issues
3. Contact support team

---

**Note**: Selalu test deployment di staging environment sebelum production!
