import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useAuth } from "../contexts/AuthContext";
import Layout from "../components/Layout";
import ImportDialog from "../components/ImportDialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/components/ui/use-toast";
import { Plus, Search } from "lucide-react";
import backend from "~backend/client";
import type { CreateKaryawanRequest } from "~backend/karyawan/create_karyawan";

export default function KaryawanPage() {
  const { user } = useAuth();
  const [search, setSearch] = useState("");
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [formData, setFormData] = useState<CreateKaryawanRequest>({
    madrasah_id: user?.madrasahId || "",
    nip: "",
    nama_lengkap: "",
    tempat_lahir: "",
    tanggal_lahir: "",
    jenis_kelamin: "",
    agama: "",
    alamat: "",
    no_telepon: "",
    email: "",
    pendidikan_terakhir: "",
    jabatan: "",
    bagian: "",
    tanggal_masuk: "",
    status_kepegawaian: "",
  });

  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data, isLoading } = useQuery({
    queryKey: ["karyawan", user?.madrasahId, search],
    queryFn: () => backend.karyawan.listKaryawan({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined,
      search, 
      limit: 100 
    }),
  });

  const createMutation = useMutation({
    mutationFn: (data: CreateKaryawanRequest) => backend.karyawan.createKaryawan(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["karyawan"] });
      setIsDialogOpen(false);
      setFormData({
        madrasah_id: user?.madrasahId || "",
        nip: "",
        nama_lengkap: "",
        tempat_lahir: "",
        tanggal_lahir: "",
        jenis_kelamin: "",
        agama: "",
        alamat: "",
        no_telepon: "",
        email: "",
        pendidikan_terakhir: "",
        jabatan: "",
        bagian: "",
        tanggal_masuk: "",
        status_kepegawaian: "",
      });
      toast({
        title: "Berhasil",
        description: "Data karyawan berhasil ditambahkan",
      });
    },
    onError: (error) => {
      console.error("Create karyawan error:", error);
      toast({
        title: "Gagal",
        description: "Gagal menambahkan data karyawan",
        variant: "destructive",
      });
    },
  });

  const importMutation = useMutation({
    mutationFn: (data: any[]) => {
      console.log("Import karyawan data received:", data);
      
      const mappedData = data.map((row, index) => {
        console.log(`Processing karyawan row ${index + 1}:`, row);
        
        return {
          nip: String(row["NIP"] || row["nip"] || "").trim(),
          nama_lengkap: String(row["Nama Lengkap"] || row["nama_lengkap"] || "").trim(),
          tempat_lahir: String(row["Tempat Lahir"] || row["tempat_lahir"] || "").trim(),
          tanggal_lahir: String(row["Tanggal Lahir"] || row["tanggal_lahir"] || "").trim(),
          jenis_kelamin: String(row["Jenis Kelamin"] || row["jenis_kelamin"] || "").trim(),
          agama: String(row["Agama"] || row["agama"] || "").trim(),
          alamat: String(row["Alamat"] || row["alamat"] || "").trim(),
          no_telepon: String(row["No Telepon"] || row["no_telepon"] || "").trim(),
          email: String(row["Email"] || row["email"] || "").trim(),
          pendidikan_terakhir: String(row["Pendidikan Terakhir"] || row["pendidikan_terakhir"] || "").trim(),
          jabatan: String(row["Jabatan"] || row["jabatan"] || "").trim(),
          bagian: String(row["Bagian"] || row["bagian"] || "").trim(),
          tanggal_masuk: String(row["Tanggal Masuk"] || row["tanggal_masuk"] || "").trim(),
          status_kepegawaian: String(row["Status Kepegawaian"] || row["status_kepegawaian"] || "").trim(),
        };
      });
      
      console.log("Mapped karyawan data:", mappedData);
      return backend.karyawan.importKaryawan({
        madrasah_id: user?.madrasahId || "",
        data: mappedData
      });
    },
    onSuccess: (result) => {
      queryClient.invalidateQueries({ queryKey: ["karyawan"] });
      toast({
        title: "Import Selesai",
        description: `Berhasil: ${result.success}, Gagal: ${result.failed}`,
      });
      if (result.errors.length > 0) {
        console.error("Import errors:", result.errors);
        toast({
          title: "Import Warnings",
          description: `Ada ${result.failed} data yang gagal diimport. Lihat console untuk detail.`,
          variant: "destructive",
        });
      }
    },
    onError: (error) => {
      console.error("Import karyawan error:", error);
      toast({
        title: "Import Gagal",
        description: "Gagal mengimport data karyawan",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    createMutation.mutate(formData);
  };

  const handleImport = async (data: any[]) => {
    console.log("handleImport karyawan called with:", data);
    await importMutation.mutateAsync(data);
  };

  const templateFields = [
    "NIP",
    "Nama Lengkap",
    "Tempat Lahir",
    "Tanggal Lahir",
    "Jenis Kelamin",
    "Agama",
    "Alamat",
    "No Telepon",
    "Email",
    "Pendidikan Terakhir",
    "Jabatan",
    "Bagian",
    "Tanggal Masuk",
    "Status Kepegawaian"
  ];

  return (
    <Layout>
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Data Karyawan</h1>
            <p className="text-gray-600 mt-2">Kelola data karyawan di madrasah</p>
          </div>
          <div className="flex space-x-2">
            <ImportDialog
              title="Karyawan"
              templateFields={templateFields}
              onImport={handleImport}
              isLoading={importMutation.isPending}
            />
            <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
              <DialogTrigger asChild>
                <Button className="bg-teal-600 hover:bg-teal-700">
                  <Plus className="h-4 w-4 mr-2" />
                  Tambah Karyawan
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-2xl">
                <DialogHeader>
                  <DialogTitle>Tambah Karyawan Baru</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="nip">NIP</Label>
                      <Input
                        id="nip"
                        value={formData.nip}
                        onChange={(e) => setFormData({ ...formData, nip: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="nama_lengkap">Nama Lengkap *</Label>
                      <Input
                        id="nama_lengkap"
                        value={formData.nama_lengkap}
                        onChange={(e) => setFormData({ ...formData, nama_lengkap: e.target.value })}
                        required
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="tempat_lahir">Tempat Lahir</Label>
                      <Input
                        id="tempat_lahir"
                        value={formData.tempat_lahir}
                        onChange={(e) => setFormData({ ...formData, tempat_lahir: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="tanggal_lahir">Tanggal Lahir</Label>
                      <Input
                        id="tanggal_lahir"
                        type="date"
                        value={formData.tanggal_lahir}
                        onChange={(e) => setFormData({ ...formData, tanggal_lahir: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="jenis_kelamin">Jenis Kelamin</Label>
                      <Input
                        id="jenis_kelamin"
                        value={formData.jenis_kelamin}
                        onChange={(e) => setFormData({ ...formData, jenis_kelamin: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="agama">Agama</Label>
                      <Input
                        id="agama"
                        value={formData.agama}
                        onChange={(e) => setFormData({ ...formData, agama: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="alamat">Alamat</Label>
                    <Input
                      id="alamat"
                      value={formData.alamat}
                      onChange={(e) => setFormData({ ...formData, alamat: e.target.value })}
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="no_telepon">No. Telepon</Label>
                      <Input
                        id="no_telepon"
                        value={formData.no_telepon}
                        onChange={(e) => setFormData({ ...formData, no_telepon: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="email">Email</Label>
                      <Input
                        id="email"
                        type="email"
                        value={formData.email}
                        onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="pendidikan_terakhir">Pendidikan Terakhir</Label>
                      <Input
                        id="pendidikan_terakhir"
                        value={formData.pendidikan_terakhir}
                        onChange={(e) => setFormData({ ...formData, pendidikan_terakhir: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="jabatan">Jabatan</Label>
                      <Input
                        id="jabatan"
                        value={formData.jabatan}
                        onChange={(e) => setFormData({ ...formData, jabatan: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-3 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="bagian">Bagian</Label>
                      <Input
                        id="bagian"
                        value={formData.bagian}
                        onChange={(e) => setFormData({ ...formData, bagian: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="tanggal_masuk">Tanggal Masuk</Label>
                      <Input
                        id="tanggal_masuk"
                        type="date"
                        value={formData.tanggal_masuk}
                        onChange={(e) => setFormData({ ...formData, tanggal_masuk: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="status_kepegawaian">Status Kepegawaian</Label>
                      <Input
                        id="status_kepegawaian"
                        value={formData.status_kepegawaian}
                        onChange={(e) => setFormData({ ...formData, status_kepegawaian: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="flex justify-end space-x-2">
                    <Button
                      type="button"
                      variant="outline"
                      onClick={() => setIsDialogOpen(false)}
                    >
                      Batal
                    </Button>
                    <Button
                      type="submit"
                      className="bg-teal-600 hover:bg-teal-700"
                      disabled={createMutation.isPending}
                    >
                      {createMutation.isPending ? "Menyimpan..." : "Simpan"}
                    </Button>
                  </div>
                </form>
              </DialogContent>
            </Dialog>
          </div>
        </div>

        <Card>
          <CardHeader>
            <div className="flex items-center space-x-2">
              <Search className="h-4 w-4 text-gray-400" />
              <Input
                placeholder="Cari karyawan..."
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                className="max-w-sm"
              />
            </div>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-teal-600 mx-auto"></div>
                <p className="text-gray-600 mt-2">Memuat data...</p>
              </div>
            ) : (
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b">
                      <th className="text-left py-3 px-4 font-medium text-gray-900">NIP</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Nama Lengkap</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Jabatan</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Bagian</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">No. Telepon</th>
                    </tr>
                  </thead>
                  <tbody>
                    {data?.karyawan.map((karyawan) => (
                      <tr key={karyawan.id} className="border-b hover:bg-gray-50">
                        <td className="py-3 px-4 text-sm">{karyawan.nip}</td>
                        <td className="py-3 px-4 text-sm font-medium">{karyawan.nama_lengkap}</td>
                        <td className="py-3 px-4 text-sm">{karyawan.jabatan}</td>
                        <td className="py-3 px-4 text-sm">{karyawan.bagian}</td>
                        <td className="py-3 px-4 text-sm">{karyawan.status_kepegawaian}</td>
                        <td className="py-3 px-4 text-sm">{karyawan.no_telepon}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
                {data?.karyawan.length === 0 && (
                  <div className="text-center py-8 text-gray-600">
                    Tidak ada data karyawan
                  </div>
                )}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </Layout>
  );
}
