import { useState } from "react";
import { useNavigate } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/components/ui/use-toast";
import { ArrowLeft } from "lucide-react";
import backend from "~backend/client";

export default function RegisterPage() {
  const navigate = useNavigate();
  const { toast } = useToast();
  const [isLoading, setIsLoading] = useState(false);
  const [formData, setFormData] = useState({
    nsm: "",
    token: "",
    password: "",
    confirm_password: "",
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (formData.password !== formData.confirm_password) {
      toast({
        title: "Error",
        description: "Password dan konfirmasi password tidak sama",
        variant: "destructive",
      });
      return;
    }

    if (formData.password.length < 6) {
      toast({
        title: "Error",
        description: "Password minimal 6 karakter",
        variant: "destructive",
      });
      return;
    }

    setIsLoading(true);
    try {
      const result = await backend.auth.registerOperator(formData);
      
      toast({
        title: "Registrasi Berhasil!",
        description: result.message,
      });

      setTimeout(() => {
        navigate("/login");
      }, 2000);
    } catch (error: any) {
      console.error("Registration error:", error);
      toast({
        title: "Registrasi Gagal",
        description: error.message || "Terjadi kesalahan saat registrasi",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-teal-50 to-green-50 flex items-center justify-center p-4">
      <Card className="w-full max-w-md">
        <CardHeader className="space-y-1">
          <div className="flex items-center justify-between">
            <Button
              variant="ghost"
              size="sm"
              onClick={() => navigate("/login")}
              className="text-gray-600 hover:text-gray-900"
            >
              <ArrowLeft className="h-4 w-4 mr-2" />
              Kembali
            </Button>
          </div>
          <div className="flex items-center justify-center mb-4">
            <div className="w-12 h-12 bg-green-600 rounded-full flex items-center justify-center relative mr-3">
              <div className="text-white text-sm font-bold text-center leading-tight">
                <div style={{ fontSize: '10px' }}>KKMI</div>
                <div style={{ fontSize: '10px' }}>XII</div>
              </div>
              <div className="absolute -top-1 left-1/2 transform -translate-x-1/2">
                <div className="w-2 h-2 bg-yellow-400 rounded-full"></div>
              </div>
            </div>
            <div>
              <CardTitle className="text-2xl font-bold">Registrasi Operator</CardTitle>
              <CardDescription className="mt-1">
                Daftarkan akun operator madrasah
              </CardDescription>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="nsm">NSM Madrasah *</Label>
              <Input
                id="nsm"
                type="text"
                placeholder="Masukkan NSM madrasah"
                value={formData.nsm}
                onChange={(e) => setFormData({ ...formData, nsm: e.target.value })}
                required
                disabled={isLoading}
              />
              <p className="text-xs text-gray-500">
                NSM adalah kode identitas madrasah Anda
              </p>
            </div>

            <div className="space-y-2">
              <Label htmlFor="token">Token Registrasi *</Label>
              <Input
                id="token"
                type="text"
                placeholder="Masukkan token dari admin"
                value={formData.token}
                onChange={(e) => setFormData({ ...formData, token: e.target.value.toUpperCase() })}
                required
                disabled={isLoading}
                className="uppercase font-mono"
              />
              <p className="text-xs text-gray-500">
                Token diberikan oleh admin madrasah
              </p>
            </div>

            <div className="space-y-2">
              <Label htmlFor="password">Password Baru *</Label>
              <Input
                id="password"
                type="password"
                placeholder="Minimal 6 karakter"
                value={formData.password}
                onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                required
                disabled={isLoading}
                minLength={6}
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="confirm_password">Konfirmasi Password *</Label>
              <Input
                id="confirm_password"
                type="password"
                placeholder="Ulangi password"
                value={formData.confirm_password}
                onChange={(e) => setFormData({ ...formData, confirm_password: e.target.value })}
                required
                disabled={isLoading}
                minLength={6}
              />
            </div>

            <Button
              type="submit"
              className="w-full bg-teal-600 hover:bg-teal-700"
              disabled={isLoading}
            >
              {isLoading ? "Mendaftar..." : "Daftar"}
            </Button>
          </form>

          <div className="mt-6 p-4 bg-blue-50 rounded-lg">
            <h4 className="text-sm font-semibold text-blue-900 mb-2">Petunjuk Registrasi:</h4>
            <ol className="text-xs text-blue-800 space-y-1 list-decimal list-inside">
              <li>Minta token registrasi dari admin superadmin</li>
              <li>Masukkan NSM madrasah Anda</li>
              <li>Masukkan token yang diberikan admin</li>
              <li>Buat password minimal 6 karakter</li>
              <li>Setelah berhasil, login dengan NSM-OP dan password Anda</li>
            </ol>
          </div>

          <div className="mt-4 text-center text-sm text-gray-600">
            Sudah punya akun?{" "}
            <button
              onClick={() => navigate("/login")}
              className="text-teal-600 hover:text-teal-700 font-medium"
            >
              Login di sini
            </button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
